<?
//////////////////////////////////////////////////////////////////////////////////////////////
// Elastico Bridge - Client
// Versione:		1.1.0
// Data:				25/11/2005
// Descrizione:	Implementazione di un client per Elastico Bridge
// -------------------------------------------------------------------------------------------
// Modifiche:		
// 
// Versione 1.1.0 
//	25/11/2005 - Aggiunta l'implementazione del comando CMD_GET_ATTACHMENT
//	24/11/2005 - Aggiunta l'implementazione del comando CMD_GET_IMAGE
// 
// Versione 1.0.0 
// 01/07/2005 - Release iniziale
//////////////////////////////////////////////////////////////////////////////////////////////

define("READ_BLOCK_SIZE",8192,true);						// dimensione blocco in lettura
define("WRITE_BLOCK_SIZE",8192,true);						// dimensione blocco in scrittura

// comandi disponibili
define("BRIDGE_CMD_SETCURAZI",0x00000001,true);			// seleziona l'azienda corrente
define("BRIDGE_CMD_QUERY",0x00000002,true);				// esegue una query di selezione
define("BRIDGE_CMD_EXECUTE",0x00000003,true);			// esegue una query di aggiornamento
define("BRIDGE_CMD_BEGINTRANS",0x00000004,true);		// apre una transazione
define("BRIDGE_CMD_COMMITTRANS",0x00000005,true);		// conferma transazione
define("BRIDGE_CMD_ROLLBACKTRANS",0x00000006,true);	// annulla transazione
define("BRIDGE_CMD_GET_IMAGE",0x00000007,true);			// scarica l'immagine di un articolo,gruppo,agente o logo dell'azienda
define("BRIDGE_CMD_SERVER_INFO",0x00000008,true);		// scarica le informazioni sul server (versione,data ecc)
define("BRIDGE_CMD_GET_ATTACHMENT",0x00000009,true);	// scarica un allegato
define("BRIDGE_CMD_DISCONNECT",0x00FFFFFF,true);		// chiude la connessione

// risultato
define("BRIDGE_RES_OK",0x00,true);							// ok
define("BRIDGE_RES_ERROR",0xFF,true);						// indica la presenza di un errore


//////////////////////////////////////////////////////////////////////////////////////////////
// definizione costanti per l'interpretazione delle 
// informazioni sulle colonne di un risultato di 
// una query di selezione
//////////////////////////////////////////////////////////////////////////////////////////////

// tipi di colonne
define("DB_BOOLEAN",1,true);
define("DB_BYTE",2,true);
define("DB_INTEGER",3,true);
define("DB_LONG",4,true);
define("DB_CURRENCY",5,true);
define("DB_SINGLE",6,true);
define("DB_DOUBLE",7,true);
define("DB_DATE",8,true);
define("DB_TEXT",10,true);
define("DB_LONGBINARY",11,true);
define("DB_MEMO",12,true);

// attributi colonna
define("DB_FIXEDFIELD",0x0001,true);
define("DB_VARIABLEFIELD",0x0002,true);
define("DB_AUTOINCRFIELD",0x0010,true);
define("DB_UPDATABLEFIELD",0x0020,true);

//////////////////////////////////////////////////////////////////////////////////////////////
// classe base con l'implementazione del protocollo di comunicazione
// tra il client ed Elastico Bridge.
//////////////////////////////////////////////////////////////////////////////////////////////
class ElasticoBridgeProtocol {
	var $remote_address;			// indirizzo ip del server
	var $remote_port;				// porta
	var $timeout;					// timeout in secondi
	var $_socket;				   // socket
   var $last_errno;           // numero ultimo errore
   var $last_errstr;          // descrizione ultimo errore
	var $connected;				// stato della connessione

   function ElasticoBridgeProtocol($remote_address,$remote_port,$timeout=10) {
      set_error_handler(array(&$this, 'error_handler'));
	   $this->remote_address = $remote_address;
		$this->remote_port = $remote_port;
		$this->timeout = $timeout;
	}

   function error_handler($n, $m, $f, $l){
      $this->last_errno = $n;
      $this->last_errstr = $m;
      //print_r(debug_backtrace());
   }

   function _clearError() {
      unset ($this->last_errno,$this->last_errstr);
   }

   // apre la connessione con il server
	function Connect() {
      $this->_socket = fsockopen($this->remote_address, $this->remote_port, $errno, $errstr, 5);
      $this->last_errno = $errno;
      $this->last_errstr = $errstr;
		$this->connected = ($this->_socket == true);
		// imposta il timeout
		if($this->connected) stream_set_timeout($this->_socket, $this->timeout);
      
		return $this->connected;
	}

	// chiude la connessione con il server
   function Close() {
		if ($this->_socket && $this->connected) {
         $this->WritePacket(BRIDGE_CMD_DISCONNECT);
		   fclose($this->_socket);
 			$this->connected = false;
         unset($this->_socket);
      }
   }

   // legge un pacchetto di dati
   function ReadPacket() {
   	// legge la lunghezza del pacchetto
   	$packet_length = $this->_readUnsignedLong();
      if ($packet_length>0) {
         // legge il corpo del pacchetto
      	$result = $this->_readData($packet_length);
         // ritorna il corpo del pacchetto
      	return $result;
      } else {
         return false;
      }
   }

   // scrive un pacchetto di dati
   function WritePacket($command,$body) {
      if (!$this->_writeUnsignedLong(strlen($body)+4)) return;
      if (!$this->_writeUnsignedLong($command)) return;
      if (strlen($body)==0) return true;

      return $this->_writeData($body);
   }

   // legge un pacchetto contenente
   // il numero ed il messaggio d'errore
   function ReadErrorPacket() {
      $err_packet = $this->ReadPacket();
		$err_info = explode(",",$err_packet,2);
		
		$this->last_errno = $err_info[0];
		$this->last_errstr = $err_info[1];
		
		return $err_info;
   }

   // legge un blocco di dati
   function _readData($data_length) {
      // azzera contatori e buffer
   	$result = "";
   	$bytes_readed = 0;
   	// legge il corpo del pacchetto
   	while (!feof($this->_socket)) {
   		$bytes_remain = $data_length - $bytes_readed;
   		$data_block = fread($this->_socket, min(READ_BLOCK_SIZE,$bytes_remain));
   		$bytes_readed += strlen($data_block);
   		$result .= $data_block;
   		// esce dal ciclo se  stato letto tutto il pacchetto
   		if($bytes_readed == $data_length) break;
      }

   	return $result;
   }

    // scrive un blocco di dati
   function _writeData($data) {
      // azzera contatori
   	$bytes_written = 0;
      $data_length = strlen($data);
      // scrive il blocco
   	while (!feof($this->_socket)) {
   		$bytes_remain = $data_length - $bytes_written;
         $block_length = min(WRITE_BLOCK_SIZE,$bytes_remain);
         $block_data = substr($data,$bytes_written,$block_length);
         if ($bytes = fwrite($this->_socket, $block_data, $block_length)) {
            $bytes_written += $bytes;
      		// esce dal ciclo se  stato scritto tutto il blocco
      		if($bytes_written == $data_length) break;
         } else {
            // esce con errore
            return false;
         }
      }

      return true;
   }

   // legge un byte
   function _readByte() {
      return ord($this->_readData(1));
   }

   // legge un intero a 16 bit
   function _readUnsignedShort() {
      $packed = $this->_readData(2);
      $unpacked = unpack("v", $unpacked);
      return $unpacked[1];
   }

   // legge un intero a 32 bit
   function _readUnsignedLong() {
      $packed = $this->_readData(4);
      $unpacked = unpack("V", $packed);
      return $unpacked[1];
   }

   // scrive un byte
   function _writeByte($value) {
      return $this->_writeData(chr($value));
   }

   // scrive un intero a 16 bit
   function _writeUnsignedShort($value) {
      return $this->_writeData(pack("v",$value));
   }

   // scrive un intero a 32 bit
   function _writeUnsignedLong($value) {
      return $this->_writeData(pack("V",$value));
   }

   function _debug($text) {
      echo $text;
   }
}

//////////////////////////////////////////////////////////////////////////////////////////////
// classe per la gestione di una connessione con Elastico Bridge
// implementa tutti i comandi che  possibile inviare al server
//////////////////////////////////////////////////////////////////////////////////////////////
class ElasticoBridgeConnection extends ElasticoBridgeProtocol {
   var $fields = array();
	
	// dati sull'ultima immagine scaricata
	var $image_data;	// contenuto del file
	var $image_name;	// nome file (temporaneo, calcolato da Elastico Bridge)
	
	// dati sull'ultimo allegato scaricato   
	var $attachment_data;	// contenuto del file
	var $attachment_name;	// nome del file (sul server)
	var $attachment_type;	// tipo
	var $attachment_id;		// codice
	var $attachment_desc;	// descrizione
	var $attachment_note;	// note
	var $attachment_row;		// numero riga

   function ElasticoBridgeConnection($remote_address,$remote_port,$timeout=10) {
      parent::ElasticoBridgeProtocol($remote_address,$remote_port,$timeout);
	}

   // seleziona l'azienda corrente il codice azienda deve
   //  corrispondere al codice azienda di Elastico 2004
   function SetCurrentAzi($cod_azi) {
      if ($this->connected) {
         if ($this->WritePacket(BRIDGE_CMD_SETCURAZI,$cod_azi)) {
            // legge il risultato
            $result = ord($this->ReadPacket());
         	if($result==BRIDGE_RES_ERROR) {
         		// c' stato un errore, legge il pacchetto con l'errore
         		$err_info = $this->ReadErrorPacket();
               return;
            }
         }
         return true;
      }
   }

   // avvia una transazione
   function BeginTransaction() {
      if ($this->connected) {
         if ($this->WritePacket(BRIDGE_CMD_BEGINTRANS)) {
            // legge il risultato
            $result = ord($this->ReadPacket());
         	if($result==BRIDGE_RES_ERROR) {
         		// c' stato un errore, legge il pacchetto con l'errore
         		$err_info = $this->ReadErrorPacket();
               return;
            }
         }
         return true;
      }
   }

   // conferma una transazione
   function CommitTransaction() {
      if ($this->connected) {
         if ($this->WritePacket(BRIDGE_CMD_COMMITTRANS)) {
            // legge il risultato
            $result = ord($this->ReadPacket());
         	if($result==BRIDGE_RES_ERROR) {
         		// c' stato un errore, legge il pacchetto con l'errore
         		$err_info = $this->ReadErrorPacket();
               return;
            }
         }
         return true;
      }
   }

   // conferma una transazione
   function RollbackTransaction() {
      if ($this->connected) {
         if ($this->WritePacket(BRIDGE_CMD_ROLLBACKTRANS)) {
            // legge il risultato
            $result = ord($this->ReadPacket());
         	if($result==BRIDGE_RES_ERROR) {
         		// c' stato un errore, legge il pacchetto con l'errore
         		$err_info = $this->ReadErrorPacket();
               return;
            }
         }
         return true;
      }
   }

   // esegue una query e legge le informazioni sulle colonne memorizzandole
   // nell'array 'Fields'
   function Query($sql_query) {
      if ($this->connected) {
         if ($this->WritePacket(BRIDGE_CMD_QUERY,$sql_query)) {
            // il comando BRIDGE_CMD_QUERY ritorna il numero di
            // colonne che compongono la query oppure il valore BRIDGE_RES_ERROR se
            // c' stato un errore
            $field_count = ord($this->ReadPacket());
         	if($field_count == BRIDGE_RES_ERROR) {
         		// c' stato un errore, legge il pacchetto con l'errore
         		$err_info = $this->ReadErrorPacket();
               return;
            }

            // legge tutti i pacchetti con le informazioni sui campi
            $this->fields = array();
            for($i=0; $i<$field_count; $i++) {
      			$field_packet = $this->ReadPacket();
      			$this->fields[] = explode(",",$field_packet);
      		}

            return true;
         }
      }
   }

   function Execute($sql_query) {
      if ($this->connected) {
         // invia la query
         if ($this->WritePacket(BRIDGE_CMD_EXECUTE,$sql_query)) {
            // legge il risultato
            $result = ord($this->ReadPacket());
         	if($result==BRIDGE_RES_ERROR) {
         		// c' stato un errore, legge il pacchetto con l'errore
         		$err_info = $this->ReadErrorPacket();
               return;
            }
            return true;
         }
      }
   }

   // legge la prossima riga della query, ritorna 'NULL'
   // se non ci sono pi righe da leggere
   function FetchRow() {
      $row = array();
      for($i=0; $i<count($this->fields); $i++) {
			if (false !== $field_value = $this->ReadPacket()) {
            if (ord($field_value)==0) {
               $row[$this->fields[$i][1]] = NULL;
            } else {
               $row[$this->fields[$i][1]] = $field_value;
            }
         } else {
            // un pacchetto nullo indica la fine delle righe
            return;
         }
		}

   	return $row;
   }

   // scarica l'immagine di un articolo,gruppo,agente o logo dell'azienda
   function GetImage($request,$filename) {
      // azzera dati sull'immagine
		unset(
			$this->image_data,
			$this->image_name
		);

		if ($this->connected) {
			if ($this->WritePacket(BRIDGE_CMD_GET_IMAGE,$request)) {
            // il comando BRIDGE_CMD_GET_IMAGE ritorna BRIDGE_RES_OK
            // se l'immagine  disponibile, altrimenti ritorna un errore
            $result = ord($this->ReadPacket());
         	if($result == BRIDGE_RES_ERROR) {
         		// c' stato un errore, legge il pacchetto con l'errore
         		$err_info = $this->ReadErrorPacket();
               return;
            }
         	// legge il nome del file
				$this->image_name = $this->ReadPacket();
				// legge il pacchetto con l'immagine
            if($this->image_data = $this->ReadPacket()) {
            	// salva l'immagine nel file se richiesto
					if(isset($filename)) {
						@unlink($filename);
						if ($handle = fopen($filename, "wb")) {
							// salva l'immagine
							fwrite($handle, $this->image_data);
							fclose($handle);
						}
					}
					return true;
            }
         }
      }
   }

   // scarica le informazioni sul server
	// il campo $flags  destinato ad uso futuro
   function ServerInfo($flags="") {
		if ($this->connected) {
			if ($this->WritePacket(BRIDGE_CMD_SERVER_INFO,$flags)) {
            // il comando BRIDGE_CMD_SERVER_INFO ritorna BRIDGE_RES_OK
            // se le informazioni sono disponibili, altrimenti ritorna un errore
            $result = ord($this->ReadPacket());
         	if($result == BRIDGE_RES_ERROR) {
         		// c' stato un errore, legge il pacchetto con l'errore
         		$err_info = $this->ReadErrorPacket();
               return;
            }
				
				$info = array();
				// versione di Elastico Bridge
				$info["bridge_release"] = $this->ReadPacket();
         	// versione di Elastico 2004
				$info["elastico_release"] = $this->ReadPacket();
         	// data e ora del server
				$info["bridge_datetime"] = $this->ReadPacket();

				return $info;
         }
      }
   }

   // scarica l'immagine di un articolo,gruppo,agente o logo dell'azienda
   function GetAttachment($request,$filename) {
		// azzera dati sull'allegato
		unset(
			$this->attachment_data,
			$this->attachment_name,
			$this->attachment_type,
			$this->attachment_id,
			$this->attachment_desc,
			$this->attachment_note,
			$this->attachment_row
		);

		if ($this->connected) {
			if ($this->WritePacket(BRIDGE_CMD_GET_ATTACHMENT,$request)) {
            // il comando BRIDGE_CMD_GET_ATTACHMENT ritorna BRIDGE_RES_OK
            // se l'allegato  disponibile, altrimenti ritorna un errore
            $result = ord($this->ReadPacket());
         	if($result == BRIDGE_RES_ERROR) {
         		// c' stato un errore, legge il pacchetto con l'errore
         		$err_info = $this->ReadErrorPacket();
               return;
            }
				
				// legge informazioni
				$this->attachment_name = $this->ReadPacket();	// nome file
				$this->attachment_type = $this->ReadPacket();	// tipo
				$this->attachment_id = $this->ReadPacket();		// codice
				$this->attachment_desc = $this->ReadPacket();	// descrizione
				$this->attachment_row = $this->ReadPacket();		// numero riga
				
				// legge il pacchetto con il contenuto del file
            if($this->attachment_data = $this->ReadPacket()) {
            	// salva l'immagine nel file se richiesto
					if(isset($filename)) {
						@unlink($filename);
						if ($handle = fopen($filename, "wb")) {
							// salva l'immagine
							fwrite($handle, $this->attachment_data);
							fclose($handle);
						}
					}
					return true;
            }
         }
      }
   }
}

?>